// ---------------------------------
// tinyBasic v1.1
//   Copyleft 2005-2007
//     by Laurent DUVEAU
//   http://www.aldweb.com/
//
// This file is a PP & NaPP for
// Palm and a FreePascal for PC
// sample Tiny Basic interpreter, 
// loosely adapted from the 
// tinyBasic made with iziBasic,
// itself loosely adapted from the 
// original Tiny Basic version by
// Li Chen Wang.
// ---------------------------------


// ---------------------------------
// Platform and Compiler choice
// ---------------------------------

//{$DEFINE Palm}  // uncomment if compiling with PP or NaPP for Palm
                  // comment if compiling with FreePascal for PC
//{$define NaPP}  // uncomment if compiling with NaPP for Palm
                  // comment if compiling with PP for Palm or FreePascal for PC


// ---------------------------------
// tinyBasic Header
// ---------------------------------

Program tinyBasic;

{$IFDEF Palm}

{$r tinyBasic.rsrc}
{$version 1.1}

{$ELSE}

uses
 Crt,DateUtils,SysUtils;
var
 TickIni:TDateTime;
 
{$ENDIF}


// ---------------------------------
// tinyBasic global variables
// ---------------------------------

const
 maxSource=99;   // 99 program lines
 nulChar=chr(0); // Char #0
  
var 
 CharIndex:Integer;  // character index in line
 ErrorLine:Integer;  // line number for error msg
 LineIndex:Integer;  // number of lines
 NextStep:Integer;   // interpreter next step
 StackIndex:Integer; // expression stack index
 VarIndex:Integer;   // variable index
 Number:Real;        // number 
 CharRead:Char;      // character
 LineStr:String;     // single statement
 ErrorStr:String;    // error message

 NumVar:array[1..26] of Real;           // 26 variables A-Z
 MathStack:array[1..30] of Real;        // 30 items math stack
 Source:array[0..maxSource] of String;  // 99 program lines (index 0 is for direct input)
 Help:array[1..10] of String;           // Help strings


{$IFDEF Palm}

{$ifndef NaPP}
// ---------------------------------
// PalmOS4 API calls
// ---------------------------------

{$i PalmAPI.pas}

function  DmSetDatabaseInfo(cardNo:UInt16;dbID:LocalID;const name:String;var attributes,version:UInt16;var crDate,modDate,bkpUpDate,modNum:UInt32;var appInfoID,sortInfoID:LocalID;var t,c:UInt32):UInt16; 
 inline(SYSTRAP,$A047);


{$else}
// ---------------------------------
// PalmOS5 API calls
// ---------------------------------

{$i ArmAPI.pas}
function  DmNumRecords(dbP:DmOpenRef):UInt16; 
 inline($E519C008,$E1A0E00F,$E59CF1DC);
function  DmSetDatabaseInfo(dbID:LocalID;const name:String;var attributes,version:UInt16;var crDate,modDate,bkpUpDate,modNum:UInt32;var appInfoID,sortInfoID:LocalID;var t,c:UInt32):UInt16; 
 inline($E519C008,$E1A0E00F,$E59CF250);
function TimGetTicks:integer;
 inline($E519C008,$E1A0E00F,$E59CF928);

{$endif}


// ---------------------------------
// Palm PP & NaPP specific functions
// ---------------------------------

procedure Dec(var MyInt:Integer);
begin
 MyInt:=MyInt-1;
end;


procedure Inc(var MyInt:Integer);
begin
 MyInt:=MyInt+1;
end;


function LowerCase(const S:Char):Char;
begin
 if (S>='A') and (S<='Z') then
  LowerCase:=chr(ord(S)+32)
 else
  LowerCase:=S;
end;


function KeyPressed:Boolean;
var
 C:UInt8;
 Event:EventType;
begin
 C:=0;
{$ifndef NaPP}
 EvtGetEvent(Event,1);
{$else}
 SysEventGet(Event,1);
{$endif}
 SysHandleEvent(Event);
 if Event.eType=keydownEvent then
{$ifndef NaPP}
  if Event.chr in [0..255] then
   C:=Event.chr;
{$else}
  if Event.keyDown.chr in [0..255] then
   C:=Event.keyDown.chr;
{$endif}
 KeyPressed:=(C<>0);
end;


function FileExists(const MyFile:String):Boolean;
begin
{$ifndef NaPP}
 FileExists:=DmFindDatabase(0,MyFile)>0;
{$else}
 FileExists:=DmFindDatabase(MyFile)>0;
{$endif}
end;


function FileErase(const MyFile:String):Boolean;
var
 IDMyFile:LocalID;
begin
{$ifndef NaPP}
 IDMyFile:=DmFindDatabase(0,MyFile);
{$else}
 IDMyFile:=DmFindDatabase(MyFile);
{$endif}
 if IDMyFile>0 then
{$ifndef NaPP}
  FileErase:=DmDeleteDatabase(0,IDMyFile)=0
{$else}
  FileErase:=DmDeleteDatabase(IDMyFile)=0
{$endif}
 else
  FileErase:=false;
end;


function FileLoad(const MyFile:String):Boolean;
const
 MyRecordStringMax=63;
type
 MyRecordString=array[0..MyRecordStringMax] of char;
 MyRecordStringP=^MyRecordString;
var
 IDMyFile:LocalID;
 RefMyFile:DmOpenRef;
 MyRecord:MyRecordString;
 MyRecordP:MyRecordStringP;
 h:MemHandle;
 NBRecords:Integer;
 i,j:UInt16;
 s:String;
 Erreur:Err;
begin
{$ifndef NaPP}
 IDMyFile:=DmFindDatabase(0,MyFile);
{$else}
 IDMyFile:=DmFindDatabase(MyFile);
{$endif}
 if IDMyFile=0 then
  Erreur:=1
 else begin
{$ifndef NaPP} 
  RefMyFile:=DmOpenDatabase(0,IDMyFile,dmModeReadOnly);
{$else}
  RefMyFile:=DmOpenDatabase(IDMyFile,dmModeReadOnly);
{$endif}
  NBRecords:=DmNumRecords(RefMyFile);
  if NBRecords>0 then   
   for i:=0 to NBRecords-1 do begin
    s:=''; 
    for j:=0 to MyRecordStringMax do
     MyRecord[j]:=nulChar;
    h:=DmQueryRecord(RefMyFile,i);
    if h<>nil then begin
     MyRecordP:=MemHandleLock(h);
     MyRecord:=MyRecordP^;
     MemHandleUnlock(h);
     j:=0;
     while (MyRecord[j]<>nulChar) and (j<63) do begin
      s:=s+MyRecord[j];
      j:=j+1;
     end;
     Source[i+1]:=s;
    end;
   end;
  DmCloseDatabase(RefMyFile);
 end;
 FileLoad:=Erreur=0;
end;


function FileSave(const MyFile:String):Boolean;
const
 dmHdrAttrBackup=8;
 t=$44415441;  // DATA
 c=$4C447442;  // LDtB
var
 IDMyFile:LocalID;
 RefMyFile:DmOpenRef;
 s:UInt32;
 h:MemHandle;
 p:MemPtr;
 j,a:UInt16;
 Erreur:Err;
begin
{$ifndef NaPP}
 IDMyFile:=DmFindDatabase(0,MyFile);
{$else}
 IDMyFile:=DmFindDatabase(MyFile);
{$endif}
 if IDMyFile>0 then
{$ifndef NaPP}
  Erreur:=DmDeleteDatabase(0,IDMyFile)
{$else}
  Erreur:=DmDeleteDatabase(IDMyFile)
{$endif}
 else
  Erreur:=0; 
 if Erreur=0 then begin
{$ifndef NaPP}
  Erreur:=DmCreateDatabase(0,MyFile,c,t,false);
{$else}
  Erreur:=DmCreateDatabase(MyFile,c,t,false);
{$endif}
  if Erreur=0 then begin
{$ifndef NaPP}
   IDMyFile:=DmFindDatabase(0,MyFile);
{$else}
   IDMyFile:=DmFindDatabase(MyFile);
{$endif}
   if IDMyFile=0 then
    Erreur:=1
   else begin
    a:=dmHdrAttrBackup;
{$ifndef NaPP}
    Erreur:=DmSetDatabaseInfo(0,IDMyFile,nil,a,nil,nil,nil,nil,nil,nil,nil,nil,nil);
{$else}
    Erreur:=DmSetDatabaseInfo(IDMyFile,nil,a,nil,nil,nil,nil,nil,nil,nil,nil,nil);
{$endif}
    if Erreur=0 then begin
{$ifndef NaPP}
     RefMyFile:=DmOpenDatabase(0,IDMyFile,dmModeWrite);
{$else}
     RefMyFile:=DmOpenDatabase(IDMyFile,dmModeWrite);
{$endif}
     for a:=1 to maxSource do 
      if Source[a]<>'' then begin
       s:=length(Source[a])+1;
       j:=a-1;
       h:=DmNewRecord(RefMyFile,j,s);
       p:=MemHandleLock(h);
       DmSet(p,0,s,0);
{$ifndef NaPP}
       DmWrite(p,0,@Source[a],s);
{$else}
       DmWrite(p,0,Source[a],s);
{$endif}
       MemHandleUnlock(h);
       DmReleaseRecord(RefMyFile,j,false);
      end;
     DmCloseDatabase(RefMyFile);
    end;
   end;
  end;
 end;
 FileSave:=Erreur=0;
end;


function Ticks:integer;
begin
 Ticks:=TimGetTicks;
end;


function TicksPerSec:integer;
begin
 TicksPerSec:=SysTicksPerSecond;
end;


function StrToReal(S:string):real;
{$ifndef NaPP}
type
 FlpDouble=record
            Hi,Lo:integer;
           end;
procedure FlpAToF(var X:FlpDouble;var St:string); inline($7400+2,$4E4F,$0305);
function d_dtof(X:FlpDouble):real; inline($7400+12,$4E4F,$0306);
var
 D:FlpDouble;
begin
 FlpAToF(D,S);
 StrToReal:=d_dtof(D);
end;
{$else}
// Routine by Pierre Brothier
type
 MDouble=record case integer of
          0:(low, high:integer);
          1:(r:real);
         end;
var
 res,d1,mult,dix:MDouble;
 sexpo,i,len:Int8;
 expo:Int16;
 si:integer;

procedure pow10(var d:real;p:integer);
var
 f:real;
 neg:boolean;
begin
 neg:=false;
 f:=10.0;
 d:=1.0;
 if p<0 then begin
  p:=-p;
  neg:=true;
 end;
 while p>0 do begin
  if p and 1<>0 then 
   d:=d*f;
  f:=f*f;
  p:=p shr 1;
 end;
 if neg then 
  d:=1.0/d;
end;

begin//Str2Double
 len:=length(s);
 res.r:=0.0;
 mult.r:=1.0;
 i:=1;
 expo:=0;
 sexpo:=1;
 si:=0;
 if s[i]='-' then begin
  si:=$80000000;
  i:=i+1;
 end;
 while (i<=len) and (s[i]>='0') and (s[i]<='9') do begin
  d1.r:=ord(s[i])-ord('0');
  res.r:=res.r*10.0+d1.r;
  i:=i+1;
 end;
 if (i<=len) and (s[i]='.') or (s[i]=',') then begin
  i:=i+1;
  while (i<=len) and (s[i]>='0') and (s[i]<='9') do begin
   d1.r:=ord(s[i])-ord('0');
   res.r:=res.r*10.0+d1.r;
   mult.r:=mult.r*10.0;
   i:=i+1;
  end;
  res.r:=res.r/mult.r;
 end;
 if (i<=len-1) and (s[i]='e') or (s[i]='E') then begin
  i:=i+1;
  if s[i]='-' then begin
   sexpo:=-1;
   i:=i+1;
  end
  else 
   if s[i]='+' then 
    i:=i+1;
  while (i<=len) and (s[i]>='0') and (s[i]<='9') do begin
   expo:=expo*10+ord(s[i])-ord('0');
   i:=i+1;
  end;
  expo:=expo*sexpo;
  pow10(d1.r,expo);
  res.r:=res.r*d1.r;
 end;
 res.high:=res.high+si;
 StrToReal:=res.r;
end;//Str2Double
{$endif}


function IntToStr(N:Integer):String;
var
 S:String;
begin
 StrIToA(S,N);
 IntToStr:=S;
end;


{$ELSE}

// ---------------------------------
// PC FreePascal specific functions
// ---------------------------------

function FileErase(const MyFile:String):Boolean;
var
 F:Text;
begin
 assign(F,MyFile);
 {$I-}
 erase(F);
 {$I+}
 FileErase:=IOresult=0;
end;


function FileLoad(const MyFile:String):Boolean;
var
 F:Text;
 i:Integer;
begin
 assign(F,MyFile);
 {$I-}
 reset(F);
 if IOresult=0 then begin
  i:=1;
  while not EOF(F) do begin
   readln(F,Source[i]);
   Inc(i);
  end;
  close(F);
 end;
 {$I+}
 FileLoad:=IOresult=0;
end;


function FileSave(const MyFile:String):Boolean;
var
 F:Text;
 i:Integer;
begin
 assign(F,MyFile);
 {$I-}
 rewrite(F);
 if IOresult=0 then begin
  for i:=1 to maxSource do
   if Source[i]<>'' then
    writeln(F,Source[i]);
  close(F);
 end;
 {$I+}
 FileSave:=IOresult=0;
end;


procedure InitTicks;
begin
 TickIni:=now;
end;


function Ticks:integer;
begin
 Ticks:=MilliSecondsBetween(now,TickIni);
end;


function TicksPerSec:integer;
begin
 TicksPerSec:=1000;
end;


function StrToReal(const S:string):real;
var
 R:Real;
 C:Word;
begin
 Val(S,R,C);
 StrToReal:=R;
end;

{$ENDIF}


// ---------------------------------
// tinyBasic starts here
// ---------------------------------

procedure InitArrays;
var
 I:Integer;
begin
 for I:=0 to maxSource do
  Source[I]:='';
 for I:=1 to 26 do
  NumVar[I]:=0; 
 for I:=1 to 30 do
  MathStack[I]:=0;
 Help[1]:='BYE, CLEAR, CLS, END';
 Help[2]:='HELP, MEM, NEW, RUN';
 Help[3]:='GOTO | LOAD | SAVE <exp>';
 Help[4]:='IF <exp> THEN <statement>';
 Help[5]:='INPUT <var>';
 Help[6]:='[LET] <var>=<exp>';
 Help[7]:='LIST [<exp>|PAUSE]';
 Help[8]:='PRINT <exp|str>[,<exp|str>][;]';
 Help[9]:='REM <any>';
 Help[10]:='Functions: TICKS, TICKSPERSEC';
end;


procedure GetChar(Lower:Boolean);
var
 TempStr:String;
begin
 TempStr:=Source[LineIndex];
 if CharIndex<=length(TempStr) then
  if Lower then
   CharRead:=LowerCase(TempStr[CharIndex])
  else
   CharRead:=TempStr[CharIndex]
 else
  CharRead:=nulChar;
end;


procedure SkipSpace;
begin
 GetChar(true);
 while CharRead=' ' do begin
  Inc(CharIndex);
  GetChar(true);
 end;
end;


procedure GetNumber;
var
 TempStr:String;
 TempInt:Integer;
begin
 SkipSpace;
 GetChar(true);
 if CharRead='-' then begin
  TempStr:='-';
  Inc(CharIndex);
  GetChar(true);
  if ((CharRead<'0') or (CharRead>'9')) and (CharRead<>'.') then
   if ErrorStr='' then
    ErrorStr:='Invalid number';
 end
 else
  TempStr:='';
 if ErrorStr='' then begin
  TempInt:=0;
  while ((CharRead>='0') and (CharRead<='9')) or (CharRead='.') do begin 
   if CharRead='.' then begin
    Inc(TempInt);
    if TempInt>1 then
     ErrorStr:='Invalid number';
   end;
   TempStr:=TempStr+CharRead;
   Inc(CharIndex);
   GetChar(true);
  end;
  if ErrorStr='' then
   Number:=StrToReal(TempStr);
 end;
end;


procedure GetLabel;
begin
 LineStr:='';
 SkipSpace;
 GetChar(true);
 if (CharRead<'a') or (CharRead>'z') then begin 
  if ErrorStr='' then
   ErrorStr:='Invalid label';
 end
 else
  while (CharRead>='a') and (CharRead<='z') do begin 
   LineStr:=LineStr+CharRead;
   Inc(CharIndex);
   GetChar(true);
  end;
end;


procedure ReturnVar;
begin
 if (length(LineStr)=1) and (LineStr[1]>='a') and (LineStr[1]<='z') then
  VarIndex:=ord(LineStr[1])-96
 else
  if ErrorStr='' then
   ErrorStr:='Variable expected';
end;


procedure GetVar;
begin
 GetLabel;
 if ErrorStr='' then
  ReturnVar;
end;
 
 
procedure GetExpression(WhereToStart:integer);

 procedure GroupExpression;
 begin
  SkipSpace;
  GetChar(true);
  case CharRead of
   '(': begin
         Inc(CharIndex);
         GetExpression(1); // BoolExpression
         SkipSpace;
         GetChar(true);
         if CharRead=')' then
          Inc(CharIndex)
         else
          if ErrorStr='' then
           ErrorStr:='Missing '')''';
        end;
    nulChar: if ErrorStr='' then
              ErrorStr:='Invalid Factor';
   else
    if ((CharRead>='0') and (CharRead<='9')) or (CharRead='-') then begin
     GetNumber;
     Inc(StackIndex);
     MathStack[StackIndex]:=Number;
    end
    else begin
     GetLabel;
     if ErrorStr='' then
      if Length(LineStr)=1 then begin
       ReturnVar;
       Inc(StackIndex);
       MathStack[StackIndex]:=NumVar[VarIndex];
      end
      else 
       if LineStr='ticks' then begin
        Inc(StackIndex);
        MathStack[StackIndex]:=Ticks;       
       end
       else if LineStr='tickspersec' then begin
        Inc(StackIndex);
        MathStack[StackIndex]:=TicksPerSec;
       end
       else
        if ErrorStr='' then
         ErrorStr:='Function expected'; 
    end;
  end;
 end;
 
 procedure MulExpression;
 var 
  TmpInt:Integer;
 begin 
  GroupExpression;
  SkipSpace;
  GetChar(true);
  repeat
   case CharRead of
    '*': begin
          Inc(CharIndex);
          GroupExpression;
          MathStack[StackIndex-1]:=MathStack[StackIndex-1]*MathStack[StackIndex];
          Dec(StackIndex);
         end;
    '/': begin
          Inc(CharIndex);
          GroupExpression;
          if MathStack[StackIndex]<>0 then begin
           MathStack[StackIndex-1]:=MathStack[StackIndex-1]/MathStack[StackIndex];
           Dec(StackIndex);
          end
          else 
           if ErrorStr='' then
            ErrorStr:='Division by zero';
         end;
    '\': begin
          Inc(CharIndex);
          GroupExpression;
          if MathStack[StackIndex]<>0 then begin           
           TmpInt:=trunc(MathStack[StackIndex-1]/MathStack[StackIndex]);
           MathStack[StackIndex-1]:=TmpInt;
           Dec(StackIndex);
          end
          else 
           if ErrorStr='' then
            ErrorStr:='Division by zero';
         end;         
   end;
   SkipSpace;
   GetChar(true);
  until (CharRead<>'*') and (CharRead<>'/') and (CharRead<>'\');
 end;
 
 procedure AddExpression;
 begin
  MulExpression;
  SkipSpace;
  GetChar(true);
  repeat
   case CharRead of
    '+': begin
          Inc(CharIndex);
          MulExpression;
          MathStack[StackIndex-1]:=MathStack[StackIndex-1]+MathStack[StackIndex];
          Dec(StackIndex);
         end;
    '-': begin
          Inc(CharIndex);
          MulExpression;
          MathStack[StackIndex-1]:=MathStack[StackIndex-1]-MathStack[StackIndex];
          Dec(StackIndex);
         end;
   end;
   SkipSpace;
   GetChar(true);
  until (CharRead<>'+') and (CharRead<>'-');
 end;

 procedure BoolExpression;
 begin
  AddExpression;
  SkipSpace;
  GetChar(true);
  repeat
   case CharRead of
    '=': begin
          Inc(CharIndex);
          AddExpression;
          MathStack[StackIndex-1]:=ord(MathStack[StackIndex-1]=MathStack[StackIndex]);
          Dec(StackIndex);
         end;
    '>': begin
          Inc(CharIndex);
          GetChar(true);
          if CharRead='=' then begin
           Inc(CharIndex);
           AddExpression;
           MathStack[StackIndex-1]:=ord(MathStack[StackIndex-1]>=MathStack[StackIndex]);
           Dec(StackIndex);          
          end
          else begin
           AddExpression;
           MathStack[StackIndex-1]:=ord(MathStack[StackIndex-1]>MathStack[StackIndex]);
           Dec(StackIndex);         
          end;
         end;
    '<': begin
          Inc(CharIndex);
          GetChar(true);
          case CharRead of
           '=': begin
                 Inc(CharIndex);
                 AddExpression;
                 MathStack[StackIndex-1]:=ord(MathStack[StackIndex-1]<=MathStack[StackIndex]);
                 Dec(StackIndex);          
               end;
           '>': begin
                 Inc(CharIndex);
                 AddExpression;
                 MathStack[StackIndex-1]:=ord(MathStack[StackIndex-1]<>MathStack[StackIndex]);
                 Dec(StackIndex);          
               end;
           else begin
            AddExpression;
            MathStack[StackIndex-1]:=ord(MathStack[StackIndex-1]<MathStack[StackIndex]);
            Dec(StackIndex);
           end;                        
          end;
    end;
   end;
   SkipSpace;
   GetChar(true);
  until (CharRead<'<') or (CharRead>'>');
 end;
 
begin
 if ErrorStr='' then
  case WhereToStart of
   0: begin
       MathStack[1]:=0;
       StackIndex:=1;
       BoolExpression;
       Number:=MathStack[StackIndex];
      end;
   1: BoolExpression;
  end;
end;


procedure EnterLine;
var
 TempInt,I:Integer;
begin
 TempInt:=trunc(Number);
 LineIndex:=1;
 CharIndex:=1;
 GetNumber;
 while (Number<TempInt) and (Number<>0) and (LineIndex<=maxSource) do begin
  Inc(LineIndex);
  CharIndex:=1;
  GetNumber;
 end;
 if LineIndex>maxSource then
  ErrorStr:='Program Overflow'
 else begin
  if Number<>TempInt then 
   for I:=maxSource downto LineIndex do 
    Source[I]:=Source[I-1];
  Source[LineIndex]:=Source[0];
  SkipSpace;
  if CharRead=nulChar then
   for I:=LineIndex to (maxSource-1) do
    Source[I]:=Source[I+1]; 
 end;
end;


procedure Interpreter;
 
 procedure ExecIf;
 begin
  GetExpression(0);
  if ErrorStr<>'' then
   NextStep:=1
  else 
   if Number<1 then begin
    CharIndex:=Length(Source[LineIndex])+1;
    NextStep:=6;
   end
   else begin
    GetLabel;
    if ErrorStr<>'' then
     NextStep:=1
    else 
     if LineStr<>'then' then begin
      ErrorStr:='''THEN'' expected';
      NextStep:=1;
     end;           
     // else NextStep:=5;
   end;        
 end;

 procedure ExecRem;
 begin
  CharIndex:=Length(Source[LineIndex])+1;
  NextStep:=6; 
 end;

 procedure ExecInput;
 var
  TempStr:String;
 begin
  GetVar;
  if ErrorStr<>'' then
   NextStep:=1
  else begin
   readln(TempStr);
   NumVar[VarIndex]:=StrToReal(TempStr);
   NextStep:=6;
  end; 
 end;

 procedure ExecPrint;
 var
  TempStr:String;
 begin
  NextStep:=0;
  while NextStep=0 do begin
   SkipSpace;
   GetChar(true); 
   if CharRead='"' then begin
    TempStr:='';
    Inc(CharIndex);
    GetChar(false);  
    while (CharRead<>'"') and (CharRead<>nulChar) do begin
     TempStr:=TempStr+CharRead;
     Inc(CharIndex);
     GetChar(false);
    end;
    if CharRead=nulChar then begin
     ErrorStr:='Unterminated string';
     NextStep:=1;
    end
    else begin
     Inc(CharIndex);
     write(TempStr);
    end;
   end
   else begin
    GetExpression(0);
    if ErrorStr<>'' then
     NextStep:=1
    else 
     if Number=trunc(Number) then
      write(trunc(Number))
     else
      write(Number);
   end;
   SkipSpace;
   GetChar(true);
   if CharRead=',' then
    Inc(CharIndex)
   else begin
    if CharRead<>';' then
     writeln
    else
     Inc(CharIndex);
    NextStep:=6; 
   end;
  end;
 end;

 procedure ExecClear;
 var
  I:Integer;
 begin
  for I:=1 to 26 do
   NumVar[I]:=0;   
  NextStep:=6; 
 end; 
  
 procedure ExecRun;
 begin
  ExecClear;
  LineIndex:=1;
  CharIndex:=1;   
  NextStep:=7; 
 end; 

   
 procedure ExecGoto;
 var
  TempInt:Integer;
 begin
  GetExpression(0);
  if ErrorStr<>'' then 
   NextStep:=1
  else begin
   TempInt:=trunc(Number);
   if ErrorLine>=TempInt then
    LineIndex:=1;
   CharIndex:=1;
   NextStep:=0;
   while NextStep=0 do begin
    if LineIndex>maxSource then begin
     ErrorStr:='Line not found';
     NextStep:=1;
    end
    else begin
     GetNumber;
     if Number=TempInt then begin
       ErrorLine:=trunc(Number);
       NextStep:=5;
     end
     else begin
      Inc(LineIndex);
      CharIndex:=1;
     end;
    end;
   end;
  end;
 end; 

 procedure ExecNew;
 var
  I:Integer;
 begin
  for I:=1 to maxSource do
   Source[I]:='';
  ExecClear;
  if ErrorLine=0 then
   NextStep:=6
  else   
   NextStep:=1; 
 end;  

 procedure ExecCls;
 begin
  ClrScr;
  NextStep:=6; 
 end; 

 procedure ExecHelp;
 var
  I:Integer; 
 begin
  for I:=1 to 10 do
   writeln(Help[I]);
  NextStep:=6; 
 end; 

 procedure ExecMem;
 var
  TempInt,I:Integer;
 begin
  TempInt:=0;
  for I:=1 to maxSource do 
   if Source[I]<>'' then
    TempInt:=I;
  writeln(maxSource-TempInt,' lines free');
  NextStep:=6; 
 end; 
         
 procedure ExecLet;
 var 
  TempInt:Integer;
 begin
  NextStep:=0;
  if LineStr='let' then begin
   GetLabel;
   if ErrorStr<>'' then
    NextStep:=1
   else
    TempInt:=1;
  end
  else
   TempInt:=0;
  if NextStep=0 then begin
   ReturnVar;
   if ErrorStr<>'' then begin
    if TempInt=0 then
     ErrorStr:='Syntax error';
    NextStep:=1;
   end
   else begin
    SkipSpace;
    GetChar(true);
    if CharRead<>'=' then begin
     ErrorStr:='''='' expected';
     NextStep:=1;
    end
    else begin
     Inc(CharIndex);
     TempInt:=VarIndex;
     GetExpression(0);
     if ErrorStr<>'' then
      NextStep:=1
     else begin 
      NumVar[TempInt]:=Number;     
      NextStep:=6;
     end;
    end;
   end; 
  end; 
 end;

 procedure ExecList;
 var 
  TempInt,TempLine,TempChar,I:Integer;
 begin
  GetNumber;
  TempInt:=trunc(Number);
  TempLine:=LineIndex;
  TempChar:=CharIndex;
  if TempInt=0 then begin
   GetLabel;
   if (ErrorStr='') and (LineStr='pause') then
    TempChar:=CharIndex;
   ErrorStr:='';
  end;
  for I:=1 to maxSource do begin
   LineIndex:=I;
   CharIndex:=1;
   GetNumber;
   if (TempInt=0) or (Number=TempInt) then 
    if Source[LineIndex]<>'' then begin
     writeln(Source[LineIndex]);
     if LineStr='pause' then 
      if (LineIndex mod 10)=0 then begin
       writeln('Pause... ');
       repeat until KeyPressed;
      end;
    end;
  end;
  TempInt:=trunc(Number);
  LineIndex:=TempLine;
  CharIndex:=TempChar;
  NextStep:=6;
 end;

 procedure ExecSave;
 var
  TempStr:String;
 begin
  GetExpression(0);
  if ErrorStr<>'' then 
   NextStep:=1
  else begin
   TempStr:='tinyBas'+IntToStr(trunc(Number));
   if Source[1]='' then
    FileErase(TempStr)
   else 
    FileSave(TempStr);
   NextStep:=6;
  end;
 end;

 procedure ExecLoad;
 var
  TempStr:String;
  I:Integer;
 begin
  GetExpression(0);
  if ErrorStr<>'' then 
   NextStep:=1
  else begin
   TempStr:='tinyBas'+IntToStr(trunc(Number));
   if not FileExists(TempStr) then begin
    ErrorStr:='File '+TempStr+' not found';
    NextStep:=1;
   end
   else begin
    for I:=1 to maxSource do
     Source[I]:='';
    FileLoad(TempStr);
    if ErrorLine=0 then
     NextStep:=6
    else
     NextStep:=1;
   end;
  end;
 end;
    
begin
 NextStep:=0;
 ErrorLine:=0;
 Number:=0;
 ErrorStr:='';
 writeln('tinyBasic v1.1');
 writeln('  Copyleft 2005-2007');
 writeln('    by Laurent DUVEAU');
 writeln('  http://www.aldweb.com/');
 writeln;
 repeat
  case NextStep of
   // Start
   0: if FileExists('tinyBas0') then begin
       Source[0]:='load0:run';
       NextStep:=3;
      end
      else
       NextStep:=1;
   // Ready
   1: begin
       if ErrorStr<>'' then begin
        if ErrorLine>0 then
         writeln('#Err in ',ErrorLine,': ',ErrorStr)
        else
         writeln('#Err: ',ErrorStr);
        ErrorStr:='';
       end;
       writeln('Ready');
       NextStep:=2;
      end;
   // Input
   2: begin
       write('> ');
       readln(Source[0]);
       NextStep:=3;
      end;
   // RunOrMemorize
   3: begin
       LineIndex:=0;
       CharIndex:=1;
       GetNumber;
       ErrorLine:=trunc(Number);
       if Number=0 then begin
        if CharRead=nulChar then
         NextStep:=1
        else
         NextStep:=5;        
       end
       else begin
        if Number=trunc(Number) then
         EnterLine
        else
         ErrorStr:='Invalid line number';
        if ErrorStr<>'' then
         NextStep:=1
        else
         NextStep:=2;
       end;  
      end;
   // Exec
   4: begin
       GetNumber;
       ErrorLine:=trunc(Number);
       NextStep:=5;
      end;
   // NextStatement
   5: if KeyPressed then begin
       writeln('Break in ',ErrorLine);
       NextStep:=1;
      end
      else begin
       GetLabel;
       if ErrorStr<>'' then
        NextStep:=1
       else
        if LineStr='if' then 
         ExecIf
        else if LineStr='rem' then
         ExecRem
        else if LineStr='input' then
         ExecInput
        else if LineStr='print' then
         ExecPrint
        else if LineStr='clear' then
         ExecClear
        else if LineStr='run' then
         ExecRun
        else if LineStr='goto' then
         ExecGoto 
        else if LineStr='new' then
         ExecNew 
        else if LineStr='cls' then
         ExecCls
        else if LineStr='help' then
         ExecHelp  
        else if LineStr='mem' then
         ExecMem
        else if LineStr='list' then
         ExecList
        else if LineStr='save' then
         ExecSave         
        else if LineStr='load' then
         ExecLoad                                 
        else if LineStr='end' then
         NextStep:=1                                            
        else if LineStr='bye' then
         NextStep:=9
        else 
         ExecLet;
      end;
   // FinishStatement
   6: begin
       SkipSpace;
       GetChar(true);
       if CharRead=':' then begin
        Inc(CharIndex);
        NextStep:=5;
       end
       else
        if CharRead<>nulChar then begin
         ErrorStr:='End of statement expected';
         NextStep:=1;         
        end
        else 
         if LineIndex=0 then
          NextStep:=1
         else begin
          Inc(LineIndex);
          CharIndex:=1;
          if LineIndex=100 then begin
           ErrorStr:='Program Overflow';
           NextStep:=1;
          end
          else
           NextStep:=7;
         end;
      end;
   // FinishStatement2
   7: if Source[LineIndex]='' then
       NextStep:=1
      else
       NextStep:=4;
  end;
 until NextStep=9;
end;


begin
{$IFNDEF Palm}
 InitTicks;
{$ENDIF}
 InitArrays;
 Interpreter;
end.
